"""Dagger IDE - A simple editor and runner for Dagger programs.

A minimal notepad-style IDE with F5 to run Dagger programs.
Uses the Rust Dagger compiler.
"""

import wx
import wx.adv
import subprocess
import tempfile
import os
import re
import sys
from pathlib import Path


def find_dagger_compiler():
    """Find the dagger compiler executable."""
    script_dir = Path(__file__).parent

    # Possible locations for the dagger compiler
    if sys.platform == "win32":
        exe_name = "dagger.exe"
    else:
        exe_name = "dagger"

    candidates = [
        script_dir / exe_name,                           # Same directory
        script_dir / "target" / "release" / exe_name,    # Release build
        script_dir / "target" / "debug" / exe_name,      # Debug build
    ]

    for path in candidates:
        if path.exists():
            return str(path)

    # Try system PATH
    try:
        result = subprocess.run(
            ["where" if sys.platform == "win32" else "which", "dagger"],
            capture_output=True,
            text=True
        )
        if result.returncode == 0:
            return result.stdout.strip().split('\n')[0]
    except:
        pass

    return None


class DaggerIDE(wx.Frame):
    """Main IDE window."""

    def __init__(self):
        super().__init__(None, title="Dagger IDE", size=(900, 700))

        self.current_file = None
        self.modified = False
        self.find_data = wx.FindReplaceData()
        self.find_dialog = None
        self.dagger_compiler = find_dagger_compiler()

        self._create_menu()
        self._create_ui()
        self._bind_events()

        self._update_title()
        self.Centre()
        self.Show()

        # Show warning if compiler not found
        if not self.dagger_compiler:
            wx.CallAfter(self._show_compiler_warning)

    def _show_compiler_warning(self):
        """Show warning about missing compiler."""
        wx.MessageBox(
            "Dagger compiler not found!\n\n"
            "Please build the compiler with:\n"
            "  cargo build --release\n\n"
            "Or place dagger.exe in the same directory as this script.",
            "Compiler Not Found",
            wx.OK | wx.ICON_WARNING
        )

    def _create_menu(self):
        """Create menu bar."""
        menubar = wx.MenuBar()

        # File menu
        file_menu = wx.Menu()
        file_menu.Append(wx.ID_NEW, "&New\tCtrl+N", "Create new file")
        file_menu.Append(wx.ID_OPEN, "&Open\tCtrl+O", "Open file")
        file_menu.Append(wx.ID_SAVE, "&Save\tCtrl+S", "Save file")
        file_menu.Append(wx.ID_SAVEAS, "Save &As...\tCtrl+Shift+S", "Save file as")
        file_menu.AppendSeparator()
        file_menu.Append(wx.ID_EXIT, "E&xit\tAlt+F4", "Exit application")
        menubar.Append(file_menu, "&File")

        # Edit menu
        edit_menu = wx.Menu()
        edit_menu.Append(wx.ID_UNDO, "&Undo\tCtrl+Z", "Undo")
        edit_menu.Append(wx.ID_REDO, "&Redo\tCtrl+Y", "Redo")
        edit_menu.AppendSeparator()
        edit_menu.Append(wx.ID_CUT, "Cu&t\tCtrl+X", "Cut")
        edit_menu.Append(wx.ID_COPY, "&Copy\tCtrl+C", "Copy")
        edit_menu.Append(wx.ID_PASTE, "&Paste\tCtrl+V", "Paste")
        edit_menu.AppendSeparator()
        edit_menu.Append(wx.ID_SELECTALL, "Select &All\tCtrl+A", "Select all")
        edit_menu.AppendSeparator()
        edit_menu.Append(wx.ID_FIND, "&Find\tCtrl+F", "Find text")
        edit_menu.Append(wx.ID_REPLACE, "&Replace\tCtrl+H", "Find and replace text")
        menubar.Append(edit_menu, "&Edit")

        # Run menu
        run_menu = wx.Menu()
        self.run_item = run_menu.Append(wx.ID_ANY, "&Run\tF5", "Run Dagger program")
        self.compile_item = run_menu.Append(wx.ID_ANY, "&Compile\tCtrl+M", "Compile to C++ and executable")
        menubar.Append(run_menu, "&Run")

        # Help menu
        help_menu = wx.Menu()
        help_menu.Append(wx.ID_ABOUT, "&About", "About Dagger IDE")
        menubar.Append(help_menu, "&Help")

        self.SetMenuBar(menubar)

    def _create_ui(self):
        """Create main UI components."""
        # Main splitter
        self.splitter = wx.SplitterWindow(self, style=wx.SP_LIVE_UPDATE)

        # Editor panel
        editor_panel = wx.Panel(self.splitter)
        editor_sizer = wx.BoxSizer(wx.VERTICAL)

        # Text editor
        self.editor = wx.TextCtrl(
            editor_panel,
            style=wx.TE_MULTILINE | wx.TE_PROCESS_TAB | wx.HSCROLL
        )
        self.editor.SetFont(wx.Font(11, wx.FONTFAMILY_MODERN, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
        editor_sizer.Add(self.editor, 1, wx.EXPAND | wx.ALL, 2)
        editor_panel.SetSizer(editor_sizer)

        # Output panel
        output_panel = wx.Panel(self.splitter)
        output_sizer = wx.BoxSizer(wx.VERTICAL)

        output_label = wx.StaticText(output_panel, label="Output:")
        output_sizer.Add(output_label, 0, wx.LEFT | wx.TOP, 5)

        self.output = wx.TextCtrl(
            output_panel,
            style=wx.TE_MULTILINE | wx.TE_READONLY | wx.HSCROLL
        )
        self.output.SetFont(wx.Font(10, wx.FONTFAMILY_MODERN, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
        self.output.SetBackgroundColour(wx.Colour(240, 240, 240))
        output_sizer.Add(self.output, 1, wx.EXPAND | wx.ALL, 2)
        output_panel.SetSizer(output_sizer)

        # Split horizontally
        self.splitter.SplitHorizontally(editor_panel, output_panel, 450)
        self.splitter.SetMinimumPaneSize(100)

        # Main sizer
        main_sizer = wx.BoxSizer(wx.VERTICAL)
        main_sizer.Add(self.splitter, 1, wx.EXPAND)
        self.SetSizer(main_sizer)

        # Status bar
        self.statusbar = self.CreateStatusBar(2)
        self.statusbar.SetStatusWidths([-1, 150])
        self.statusbar.SetStatusText("Ready", 0)
        self.statusbar.SetStatusText("Line: 1, Col: 1", 1)

    def _bind_events(self):
        """Bind event handlers."""
        # Menu events
        self.Bind(wx.EVT_MENU, self.on_new, id=wx.ID_NEW)
        self.Bind(wx.EVT_MENU, self.on_open, id=wx.ID_OPEN)
        self.Bind(wx.EVT_MENU, self.on_save, id=wx.ID_SAVE)
        self.Bind(wx.EVT_MENU, self.on_save_as, id=wx.ID_SAVEAS)
        self.Bind(wx.EVT_MENU, self.on_exit, id=wx.ID_EXIT)

        self.Bind(wx.EVT_MENU, self.on_undo, id=wx.ID_UNDO)
        self.Bind(wx.EVT_MENU, self.on_redo, id=wx.ID_REDO)
        self.Bind(wx.EVT_MENU, self.on_cut, id=wx.ID_CUT)
        self.Bind(wx.EVT_MENU, self.on_copy, id=wx.ID_COPY)
        self.Bind(wx.EVT_MENU, self.on_paste, id=wx.ID_PASTE)
        self.Bind(wx.EVT_MENU, self.on_select_all, id=wx.ID_SELECTALL)
        self.Bind(wx.EVT_MENU, self.on_find, id=wx.ID_FIND)
        self.Bind(wx.EVT_MENU, self.on_replace, id=wx.ID_REPLACE)
        self.Bind(wx.EVT_FIND, self.on_find_next)
        self.Bind(wx.EVT_FIND_NEXT, self.on_find_next)
        self.Bind(wx.EVT_FIND_REPLACE, self.on_find_replace)
        self.Bind(wx.EVT_FIND_REPLACE_ALL, self.on_find_replace_all)
        self.Bind(wx.EVT_FIND_CLOSE, self.on_find_close)

        self.Bind(wx.EVT_MENU, self.on_run, self.run_item)
        self.Bind(wx.EVT_MENU, self.on_compile, self.compile_item)
        self.Bind(wx.EVT_MENU, self.on_about, id=wx.ID_ABOUT)

        # Editor events
        self.editor.Bind(wx.EVT_TEXT, self.on_text_changed)
        self.editor.Bind(wx.EVT_KEY_UP, self.on_key_up)

        # Window close
        self.Bind(wx.EVT_CLOSE, self.on_close)

    def _update_title(self):
        """Update window title."""
        name = self.current_file if self.current_file else "Untitled"
        modified = " *" if self.modified else ""
        self.SetTitle(f"{name}{modified} - Dagger")

    def _update_status_position(self):
        """Update cursor position in status bar."""
        pos = self.editor.GetInsertionPoint()
        text = self.editor.GetValue()[:pos]
        line = text.count('\n') + 1
        col = pos - text.rfind('\n')
        self.statusbar.SetStatusText(f"Line: {line}, Col: {col}", 1)

    def _check_save(self):
        """Check if user wants to save modified file. Returns True to proceed."""
        if not self.modified:
            return True

        dlg = wx.MessageDialog(
            self,
            "Do you want to save changes?",
            "Unsaved Changes",
            wx.YES_NO | wx.CANCEL | wx.ICON_QUESTION
        )
        result = dlg.ShowModal()
        dlg.Destroy()

        if result == wx.ID_YES:
            return self.on_save(None)
        elif result == wx.ID_NO:
            return True
        else:
            return False

    # Event handlers
    def on_new(self, event):
        """Create new file."""
        if not self._check_save():
            return
        self.editor.Clear()
        self.current_file = None
        self.modified = False
        self._update_title()
        self.statusbar.SetStatusText("New file", 0)

    def on_open(self, event):
        """Open file."""
        if not self._check_save():
            return

        dlg = wx.FileDialog(
            self,
            "Open Dagger File",
            wildcard="Dagger files (*.dg)|*.dg|All files (*.*)|*.*",
            style=wx.FD_OPEN | wx.FD_FILE_MUST_EXIST
        )

        if dlg.ShowModal() == wx.ID_OK:
            path = dlg.GetPath()
            try:
                with open(path, 'r') as f:
                    self.editor.SetValue(f.read())
                self.current_file = path
                self.modified = False
                self._update_title()
                self.statusbar.SetStatusText(f"Opened: {path}", 0)
            except Exception as e:
                wx.MessageBox(f"Error opening file: {e}", "Error", wx.OK | wx.ICON_ERROR)

        dlg.Destroy()

    def on_save(self, event):
        """Save file."""
        if self.current_file:
            try:
                with open(self.current_file, 'w') as f:
                    f.write(self.editor.GetValue())
                self.modified = False
                self._update_title()
                self.statusbar.SetStatusText(f"Saved: {self.current_file}", 0)
                return True
            except Exception as e:
                wx.MessageBox(f"Error saving file: {e}", "Error", wx.OK | wx.ICON_ERROR)
                return False
        else:
            return self.on_save_as(event)

    def on_save_as(self, event):
        """Save file as."""
        dlg = wx.FileDialog(
            self,
            "Save Dagger File",
            wildcard="Dagger files (*.dg)|*.dg|All files (*.*)|*.*",
            style=wx.FD_SAVE | wx.FD_OVERWRITE_PROMPT
        )

        if dlg.ShowModal() == wx.ID_OK:
            path = dlg.GetPath()
            if not path.endswith('.dg'):
                path += '.dg'
            try:
                with open(path, 'w') as f:
                    f.write(self.editor.GetValue())
                self.current_file = path
                self.modified = False
                self._update_title()
                self.statusbar.SetStatusText(f"Saved: {path}", 0)
                dlg.Destroy()
                return True
            except Exception as e:
                wx.MessageBox(f"Error saving file: {e}", "Error", wx.OK | wx.ICON_ERROR)

        dlg.Destroy()
        return False

    def on_exit(self, event):
        """Exit application."""
        self.Close()

    def on_undo(self, event):
        self.editor.Undo()

    def on_redo(self, event):
        self.editor.Redo()

    def on_cut(self, event):
        self.editor.Cut()

    def on_copy(self, event):
        self.editor.Copy()

    def on_paste(self, event):
        self.editor.Paste()

    def on_select_all(self, event):
        self.editor.SelectAll()

    def on_find(self, event):
        """Open find dialog."""
        if self.find_dialog:
            self.find_dialog.Destroy()
        self.find_dialog = wx.FindReplaceDialog(
            self, self.find_data, "Find",
            wx.FR_NOWHOLEWORD
        )
        self.find_dialog.Show()

    def on_replace(self, event):
        """Open find and replace dialog."""
        if self.find_dialog:
            self.find_dialog.Destroy()
        self.find_dialog = wx.FindReplaceDialog(
            self, self.find_data, "Find and Replace",
            wx.FR_REPLACEDIALOG | wx.FR_NOWHOLEWORD
        )
        self.find_dialog.Show()

    def on_find_next(self, event):
        """Find next occurrence."""
        find_string = self.find_data.GetFindString()
        if not find_string:
            return

        flags = self.find_data.GetFlags()
        text = self.editor.GetValue()
        start_pos = self.editor.GetInsertionPoint()

        # Determine search direction and case sensitivity
        search_down = flags & wx.FR_DOWN
        match_case = flags & wx.FR_MATCHCASE

        if not match_case:
            search_text = text.lower()
            find_string_search = find_string.lower()
        else:
            search_text = text
            find_string_search = find_string

        if search_down:
            pos = search_text.find(find_string_search, start_pos)
            if pos == -1:  # Wrap around
                pos = search_text.find(find_string_search, 0)
        else:
            pos = search_text.rfind(find_string_search, 0, start_pos)
            if pos == -1:  # Wrap around
                pos = search_text.rfind(find_string_search)

        if pos != -1:
            self.editor.SetSelection(pos, pos + len(find_string))
            self.editor.SetFocus()
            self.statusbar.SetStatusText(f"Found: '{find_string}'", 0)
        else:
            wx.MessageBox(f"Cannot find '{find_string}'", "Find", wx.OK | wx.ICON_INFORMATION)
            self.statusbar.SetStatusText("Not found", 0)

    def on_find_replace(self, event):
        """Replace current selection and find next."""
        find_string = self.find_data.GetFindString()
        replace_string = self.find_data.GetReplaceString()

        if not find_string:
            return

        # Check if current selection matches find string
        selection = self.editor.GetStringSelection()
        flags = self.find_data.GetFlags()
        match_case = flags & wx.FR_MATCHCASE

        if match_case:
            matches = selection == find_string
        else:
            matches = selection.lower() == find_string.lower()

        if matches:
            # Replace the selection
            self.editor.WriteText(replace_string)
            self.statusbar.SetStatusText(f"Replaced '{find_string}' with '{replace_string}'", 0)

        # Find next occurrence
        self.on_find_next(event)

    def on_find_replace_all(self, event):
        """Replace all occurrences."""
        find_string = self.find_data.GetFindString()
        replace_string = self.find_data.GetReplaceString()

        if not find_string:
            return

        flags = self.find_data.GetFlags()
        match_case = flags & wx.FR_MATCHCASE

        text = self.editor.GetValue()
        if match_case:
            count = text.count(find_string)
            new_text = text.replace(find_string, replace_string)
        else:
            # Case-insensitive replace
            pattern = re.compile(re.escape(find_string), re.IGNORECASE)
            count = len(pattern.findall(text))
            new_text = pattern.sub(replace_string, text)

        if count > 0:
            self.editor.SetValue(new_text)
            self.statusbar.SetStatusText(f"Replaced {count} occurrence(s)", 0)
            wx.MessageBox(f"Replaced {count} occurrence(s)", "Replace All", wx.OK | wx.ICON_INFORMATION)
        else:
            wx.MessageBox(f"Cannot find '{find_string}'", "Replace All", wx.OK | wx.ICON_INFORMATION)
            self.statusbar.SetStatusText("Not found", 0)

    def on_find_close(self, event):
        """Handle find dialog close."""
        if self.find_dialog:
            self.find_dialog.Destroy()
            self.find_dialog = None

    def on_run(self, event):
        """Run the Dagger program using the Rust compiler."""
        self.output.Clear()
        self.output.SetFocus()
        self.statusbar.SetStatusText("Running...", 0)
        wx.Yield()  # Update UI

        # Check for compiler
        if not self.dagger_compiler:
            self.dagger_compiler = find_dagger_compiler()
            if not self.dagger_compiler:
                self.output.AppendText("Error: Dagger compiler not found!\n\n")
                self.output.AppendText("Please build the compiler with:\n")
                self.output.AppendText("  cargo build --release\n\n")
                self.output.AppendText("Or place dagger.exe in the same directory as this script.\n")
                self.statusbar.SetStatusText("Compiler not found", 0)
                return

        source = self.editor.GetValue()
        if not source.strip():
            self.output.AppendText("Error: No code to run\n")
            self.statusbar.SetStatusText("Ready", 0)
            return

        # Determine source directory for import resolution
        if self.current_file:
            abs_path = os.path.abspath(self.current_file)
            source_dir = os.path.dirname(abs_path)
        else:
            source_dir = os.getcwd()

        # Create temporary files in source directory so imports work
        src_path = os.path.join(source_dir, "_temp_run.dg")
        cpp_path = os.path.join(source_dir, "_temp_run.cpp")
        exe_path = os.path.join(source_dir, "_temp_run")
        if sys.platform == "win32":
            exe_path += ".exe"

        try:
            # Write source
            with open(src_path, 'w') as f:
                f.write(source)

            # Compile to C++ using Rust dagger compiler
            result = subprocess.run(
                [self.dagger_compiler, src_path, "-o", cpp_path, "-c"],
                capture_output=True,
                text=True,
                timeout=30,
                cwd=source_dir
            )

            if result.returncode != 0:
                self.output.AppendText("=== Compilation Error ===\n")
                self.output.AppendText(result.stdout)
                self.output.AppendText(result.stderr)
                self.statusbar.SetStatusText("Compilation failed", 0)
                return

            self.output.AppendText("=== Compiled to C++ ===\n")

            # Check for C++ compiler
            cpp_compiler = None
            for compiler in ["clang++", "g++", "clang++.exe", "g++.exe"]:
                try:
                    subprocess.run([compiler, "--version"], capture_output=True)
                    cpp_compiler = compiler
                    break
                except FileNotFoundError:
                    continue

            if not cpp_compiler:
                self.output.AppendText("\nNote: No C++ compiler found.\n")
                self.output.AppendText("Install clang++ or g++ to run programs.\n")
                self.output.AppendText("\n=== Generated C++ ===\n")
                try:
                    with open(cpp_path, 'r') as f:
                        self.output.AppendText(f.read())
                except:
                    pass
                self.statusbar.SetStatusText("C++ generated (no compiler)", 0)
                return

            # Compile C++ to executable
            result = subprocess.run(
                [cpp_compiler, "-std=c++17", "-O2", "-o", exe_path, cpp_path],
                capture_output=True,
                text=True,
                timeout=30
            )

            if result.returncode != 0:
                self.output.AppendText("=== C++ Compilation Error ===\n")
                self.output.AppendText(result.stderr)
                self.statusbar.SetStatusText("C++ compilation failed", 0)
                return

            # Run executable
            self.output.AppendText("=== Program Output ===\n")
            result = subprocess.run(
                [exe_path],
                capture_output=True,
                text=True,
                timeout=10
            )

            self.output.AppendText(result.stdout)
            if result.stderr:
                self.output.AppendText("\n=== Errors ===\n")
                self.output.AppendText(result.stderr)

            self.output.AppendText(f"\n=== Exit code: {result.returncode} ===\n")
            self.statusbar.SetStatusText("Run complete", 0)

        except subprocess.TimeoutExpired:
            self.output.AppendText("Error: Operation timed out\n")
            self.statusbar.SetStatusText("Timeout", 0)
        except Exception as e:
            self.output.AppendText(f"Error: {e}\n")
            self.statusbar.SetStatusText("Error", 0)
        finally:
            # Clean up temp files
            for f in [src_path, cpp_path, exe_path]:
                try:
                    os.unlink(f)
                except:
                    pass

    def on_compile(self, event):
        """Compile the Dagger program to C++ and executable in the project directory."""
        self.output.Clear()
        self.statusbar.SetStatusText("Compiling...", 0)
        wx.Yield()

        # Check for compiler
        if not self.dagger_compiler:
            self.dagger_compiler = find_dagger_compiler()
            if not self.dagger_compiler:
                self.output.AppendText("Error: Dagger compiler not found!\n\n")
                self.output.AppendText("Please build the compiler with:\n")
                self.output.AppendText("  cargo build --release\n\n")
                self.output.AppendText("Or place dagger.exe in the same directory as this script.\n")
                self.statusbar.SetStatusText("Compiler not found", 0)
                return

        source = self.editor.GetValue()
        if not source.strip():
            self.output.AppendText("Error: No code to compile\n")
            self.statusbar.SetStatusText("Ready", 0)
            return

        # Determine output directory
        if self.current_file:
            # Get absolute path to handle relative paths correctly
            abs_path = os.path.abspath(self.current_file)
            output_dir = os.path.dirname(abs_path)
            base_name = os.path.splitext(os.path.basename(abs_path))[0]
        else:
            output_dir = os.getcwd()
            base_name = "program"

        # Ensure output_dir is valid
        if not output_dir:
            output_dir = os.getcwd()

        # Create temporary source file in the same directory as the project
        # so that imports can be resolved correctly
        temp_name = f"_temp_compile_{base_name}.dg"
        src_path = os.path.join(output_dir, temp_name)

        try:
            # Write temp source file
            with open(src_path, 'w') as f:
                f.write(source)

            # Output paths in project directory
            output_base = os.path.join(output_dir, base_name)
            cpp_path = output_base + ".cpp"
            exe_path = output_base + (".exe" if sys.platform == "win32" else "")

            # Compile using Dagger compiler (without -c flag to get executable)
            self.output.AppendText(f"Source file: {src_path}\n")
            self.output.AppendText(f"Output base: {output_base}\n")
            self.output.AppendText(f"Compiler: {self.dagger_compiler}\n")
            wx.Yield()

            result = subprocess.run(
                [self.dagger_compiler, src_path, "-o", output_base],
                capture_output=True,
                text=True,
                timeout=30,
                cwd=output_dir  # Run compiler from the source directory
            )

            # Parse compile time from output
            compile_time_ms = 0
            for line in result.stdout.split('\n'):
                if line.startswith('compile_time_ms:'):
                    try:
                        compile_time_ms = int(line.split(':')[1])
                    except ValueError:
                        pass

            if result.returncode != 0:
                self.output.AppendText("=== Compilation Error ===\n")
                self.output.AppendText(result.stdout)
                self.output.AppendText(result.stderr)
                self.statusbar.SetStatusText("Compilation failed", 0)
                return

            # Show output
            self.output.AppendText("=== Compilation Successful ===\n")
            self.output.AppendText(f"Output directory: {output_dir}\n")
            if os.path.exists(cpp_path):
                self.output.AppendText(f"C++ output: {cpp_path}\n")
            if os.path.exists(exe_path):
                self.output.AppendText(f"Executable: {exe_path}\n")
            self.output.AppendText(result.stdout)

            self.statusbar.SetStatusText("Compilation complete", 0)

            # Show compile time modal
            wx.MessageBox(
                f"Compilation completed in {compile_time_ms} ms",
                "Compile Complete",
                wx.OK | wx.ICON_INFORMATION
            )

        except subprocess.TimeoutExpired:
            self.output.AppendText("Error: Compilation timed out\n")
            self.statusbar.SetStatusText("Timeout", 0)
        except Exception as e:
            self.output.AppendText(f"Error: {e}\n")
            self.statusbar.SetStatusText("Error", 0)
        finally:
            # Clean up temp file
            try:
                os.unlink(src_path)
            except:
                pass

    def on_about(self, event):
        """Show about dialog."""
        info = wx.adv.AboutDialogInfo()
        info.SetName("Dagger")
        info.SetVersion("2.0")
        info.SetDescription(
            "IDE for the Dagger programming language.\n\n"
            "Press F5 to run your program.\n\n"
            "Using Rust-based Dagger compiler."
        )
        info.SetCopyright("(C) 2024")
        wx.adv.AboutBox(info)

    def on_text_changed(self, event):
        """Handle text changes."""
        if not self.modified:
            self.modified = True
            self._update_title()
        event.Skip()

    def on_key_up(self, event):
        """Handle key up events."""
        self._update_status_position()
        event.Skip()

    def on_close(self, event):
        """Handle window close."""
        if self._check_save():
            event.Skip()
        else:
            event.Veto()


def main():
    """Main entry point."""
    app = wx.App()
    frame = DaggerIDE()

    # Load file from command line if provided
    if len(sys.argv) > 1:
        path = sys.argv[1]
        if os.path.exists(path):
            try:
                with open(path, 'r') as f:
                    frame.editor.SetValue(f.read())
                frame.current_file = path
                frame.modified = False
                frame._update_title()
            except Exception as e:
                wx.MessageBox(f"Error opening file: {e}", "Error", wx.OK | wx.ICON_ERROR)

    app.MainLoop()


if __name__ == "__main__":
    main()
